/* eslint-disable @typescript-eslint/no-explicit-any */
import React, {
  ReactSVGElement,
  SetStateAction,
  useEffect,
  useRef,
  useState,
} from "react";
import styles from "./styles/selectDropdown.module.scss";
import {
  Box,
  CircularProgress,
  Divider,
  Fade,
  IconButton,
  InputBase,
  List,
  Popper,
} from "@mui/material";
import { Close, KeyboardArrowDown } from "@mui/icons-material";
import diacriticSensitiveRegex from "@/lib/utils/diacritic-sensitive-regex";
import { ellipsisText } from "@/lib/utils/ellipsis-text";
import { useTranslations } from "next-intl";

interface Option {
  label: string;
  value: string | number;
  name: string;
  divisor?: boolean;
}

export interface CustomSelectorProps {
  options: Option[];
  placeholder: string;
  label?: string;
  selectedItem: string | number | null;
  selectedItems?: (string | number | null)[];
  onClickItem?: (
    value: string | number | undefined | (string | number | undefined)[]
  ) => void;
  onClick?: () => void;
  forceAnchorDelete?: boolean;
  allowSearchInput?: boolean;
  loading?: boolean;
  clickToRestore?: () => void; // no se utiliza en el código, pero se declara
  disabled?: boolean;
  obligatory?: boolean;
  multiple?: boolean;
  onlyDropdown?: boolean;
  style?: React.CSSProperties;
  iconEnd?: ReactSVGElement | any;
  openOnlyDropdown?: React.MouseEvent<HTMLDivElement>["currentTarget"] | null;
  setOpenDropdown?: React.Dispatch<
    SetStateAction<React.MouseEvent<HTMLDivElement>["currentTarget"] | null>
  >;
}

const CustomSelector: React.FC<CustomSelectorProps> = ({
  options,
  placeholder,
  label,
  selectedItem,
  selectedItems,
  onClickItem,
  forceAnchorDelete,
  allowSearchInput,
  loading,
  disabled,
  obligatory = false,
  onlyDropdown,
  openOnlyDropdown,
  multiple,
  setOpenDropdown,
}) => {
  const t = useTranslations();
  const [anchorEl, setAnchorEl] = useState<HTMLElement | null>(null);
  const [searchState, setSearchState] = useState<Option[]>([]);
  const [labels, setLabel] = useState<string>("");

  const [dynamicWidth, setDynamicWidth] = useState(0);

  const buttonDiv = useRef<HTMLDivElement>(null);

  const open = Boolean(anchorEl);

  const handleClickButton = (event: React.MouseEvent<HTMLDivElement>) => {
    setAnchorEl(anchorEl ? null : event.currentTarget);
  };

  const maskClickFunction = (value: string | number) => {
    if (multiple) {
      setAnchorEl(null);
      if (onClickItem) {
        let payload = [
          ...(selectedItems?.filter((item) => item !== null) || []),
        ];

        if (!selectedItems?.includes(value)) {
          payload.push(value);
        } else {
          payload = payload.filter((d) => d !== value);
        }

        onClickItem(payload);
      }
    } else {
      setAnchorEl(null);
      if (onClickItem) onClickItem(value);
    }
  };

  useEffect(() => {
    if (forceAnchorDelete) setAnchorEl(null);
  }, [forceAnchorDelete]);

  useEffect(() => {
    if (openOnlyDropdown || open) {
      setDynamicWidth(
        openOnlyDropdown
          ? (openOnlyDropdown as unknown as HTMLElement).offsetWidth
          : open
          ? (buttonDiv.current?.offsetWidth as number)
          : 200
      );
    }
  }, [openOnlyDropdown, open]);

  const handleSearch = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { value } = e.target;

    if (value.length > 0) {
      const filtration = options.filter(({ name }) => {
        const valueRegex = new RegExp(diacriticSensitiveRegex(value), "i");
        return (
          valueRegex.test(name) ||
          diacriticSensitiveRegex(name).includes(diacriticSensitiveRegex(value))
        );
      });
      setSearchState(filtration);
    } else {
      setSearchState(options);
    }
  };

  const handleClear = () => {
    setAnchorEl(null);
    if (onClickItem) onClickItem(undefined);
    if (setOpenDropdown) {
      setOpenDropdown(null);
    }
  };

  useEffect(() => {
    if (selectedItem !== null && selectedItem !== undefined) {
      const text = options?.find(
        ({ value }) =>
          value ==
          (typeof selectedItem === "number"
            ? selectedItem
            : selectedItem.toString())
      )?.label;
      setLabel(text ?? "");
    }
  }, [selectedItem, options]);

  return (
    <>
      {(open || openOnlyDropdown) && (
        <div
          onClick={() => {
            setAnchorEl(null);
            if (setOpenDropdown) setOpenDropdown(null);
          }}
          className={styles.dismissArea}
        />
      )}
      {!onlyDropdown && (
        <div
          className={styles.parentContainer}
          style={{ opacity: disabled ? "0.8" : "1" }}
        >
          {label && (
            <p className={styles.label}>
              {t(label)} {obligatory && <span className="text-danger">*</span>}
            </p>
          )}
          <div
            ref={buttonDiv}
            onClick={disabled ? undefined : handleClickButton}
            className={styles.inputMask}
          >
            <p className={labels ? styles.value : styles.placeholder}>
              {ellipsisText(
                t(labels && selectedItem ? labels : placeholder ?? ""),
                32
              )}
            </p>
            <div className={styles.buttons}>
              <span>
                {(selectedItem || selectedItem === 0) && (
                  <IconButton
                    className={styles.iconButtonClose}
                    size="small"
                    color="primary"
                    onClick={handleClear}
                  >
                    <Close sx={{ fontSize: "13px" }} />
                  </IconButton>
                )}
              </span>
              <div className={styles.loaderAndArrow}>
                {loading && <CircularProgress color="primary" size={12} />}
                <KeyboardArrowDown
                  style={{ transform: `rotateZ(${open ? "180deg" : "0deg"})` }}
                  className={styles.icon}
                  color="primary"
                />
              </div>
            </div>
          </div>
        </div>
      )}
      <Popper
        open={onlyDropdown ? Boolean(openOnlyDropdown) : open}
        style={{ zIndex: 10000 }}
        anchorEl={
          onlyDropdown && openOnlyDropdown
            ? (openOnlyDropdown as unknown as HTMLElement)
            : anchorEl
        }
        transition
      >
        {({ TransitionProps }) => (
          <Fade {...TransitionProps} timeout={350}>
            <Box
              style={{
                width: dynamicWidth - 2,
              }}
              className={styles.popperBox}
            >
              {loading && (
                <div
                  className={styles.formRow}
                  style={{
                    padding: "10px 10px 0 10px",
                    justifyContent: "center",
                    display: "flex",
                  }}
                >
                  <CircularProgress size={"22px"} color="primary" />
                </div>
              )}
              {allowSearchInput && (
                <div style={{ width: "100%", background: "red" }}>
                  <InputBase
                    className={styles.input}
                    id="search"
                    name="search"
                    placeholder={t("search")}
                    type="text"
                    onChange={handleSearch}
                  />
                </div>
              )}
              <List style={{ padding: 0, marginBottom: 0 }}>
                {options && options.length > 0 && (
                  <>
                    {(searchState.length ? searchState : options).map(
                      (option) => {
                        if (option.divisor) {
                          return <Divider key={option.label} />;
                        }
                        return (
                          <div
                            key={option.label}
                            className={
                              (
                                multiple
                                  ? selectedItems?.includes(option.value)
                                  : selectedItem === option.value
                              )
                                ? styles.listItem__selected
                                : styles.listItem
                            }
                            onClick={() => maskClickFunction(option.value)}
                          >
                            <p className={styles.text}>{t(option.label)}</p>
                          </div>
                        );
                      }
                    )}
                  </>
                )}
              </List>
            </Box>
          </Fade>
        )}
      </Popper>
    </>
  );
};

export default CustomSelector;
