import { useEffect, useState } from "react";
import { UseCountries } from "./useCountries.d";
import { useAppSelector } from "@/lib/services/redux/store";
import useRequestHandler from "../useRequestHandler/useRequestHandler";
import {
  GetCitiesParams,
  GetCitiesResponse,
  GetCountriesParams,
  GetCountriesResponse,
  GetCountryPhoneCodesParams,
  GetCountryPhoneCodesResponse,
  GetStatesParams,
  GetStatesResponse,
} from "@/lib/services/redux/reducers/app/appActions.d";
import {
  getCities,
  getCountries,
  getCountryPhoneCodes,
  getStates,
} from "@/lib/services/redux/reducers/app/appActions";
import { SelectOptionsProps } from "../../components/FormInput/types";

const useCountries = (): UseCountries => {
  const [countriesMapped, setCountriesMapped] = useState<SelectOptionsProps[]>(
    []
  );
  const [phoneCodesMapped, setPhoneCodesMapped] = useState<
    SelectOptionsProps[]
  >([]);

  const {
    getCountries: { countries },
    getCountryPhoneCodes: { countryPhoneCodes },
  } = useAppSelector((state) => state.app);

  const { fetcher: getCountriesFetch, requestStatus: countriesStatus } =
    useRequestHandler<GetCountriesResponse, GetCountriesParams>(getCountries);

  const { fetcher: getPhoneCodes, requestStatus: phoneCodesStatus } =
    useRequestHandler<GetCountryPhoneCodesResponse, GetCountryPhoneCodesParams>(
      getCountryPhoneCodes
    );

  const { fetcher: getStatesFetch, requestStatus: getStatesStatus } =
    useRequestHandler<GetStatesResponse, GetStatesParams>(getStates);

  const { fetcher: getCitiesFetch, requestStatus: getCitiesStatus } =
    useRequestHandler<GetCitiesResponse, GetCitiesParams>(getCities);

  useEffect(() => {
    if (countriesStatus.error) {
      console.error("error-getting-countries: ", countriesStatus.error);
    }
    if (phoneCodesStatus.error) {
      console.error("error-getting-phone-codes: ", phoneCodesStatus.error);
    }
  }, [phoneCodesStatus, countriesStatus]);

  useEffect(() => {
    if (countries) {
      const mapped = countries.map((country) => {
        return {
          label: country.name,
          name: country.name,
          value: country.id,
          lat: Number(country.latitude),
          lng: Number(country.longitude),
        };
      });

      setCountriesMapped(mapped);
    }
  }, [countries]);

  useEffect(() => {
    if (Array.isArray(countryPhoneCodes)) {
      setPhoneCodesMapped(
        countryPhoneCodes?.map((d) => {
          return {
            name: d.name,
            value: d.identification.toString(),
            code: d.phonecode.toString(),
            label: "+" + d.phonecode,
          };
        }) ?? []
      );
    }
  }, [countryPhoneCodes]);

  const refetchCountries = () => {
    getCountriesFetch({});
  };
  const refetchCountryPhoneCodes = () => {
    getPhoneCodes({});
  };

  const fetchStates = async (country: string) => {
    const states = await getStatesFetch({
      country,
    });

    if (states.status === "success") {
      const mapped = states.response.states.map((state) => {
        return {
          label: state.name,
          name: state.name,
          value: state.id,
          lat: state.latitude,
          lng: state.longitude,
        };
      });

      return mapped;
    } else {
      return [];
    }
  };

  const fetchCities = async (country: string, state: string) => {
    const cities = await getCitiesFetch({
      country,
      state,
    });

    if (cities.status === "success") {
      const mapped = cities.response.cities.map((city) => {
        return {
          label: city.name,
          name: city.name,
          value: city.id,
          lat: city.latitude,
          lng: city.longitude,
        };
      });

      return mapped;
    } else {
      return [];
    }
  };

  return {
    refetchCountries,
    refetchCountryPhoneCodes,
    countries: countriesMapped,
    phoneCodes: phoneCodesMapped,
    fetchStates,
    fetchCities,
    statesLoading: getStatesStatus.loading,
    citiesLoading: getCitiesStatus.loading,
  };
};

export default useCountries;
