/* eslint-disable @typescript-eslint/no-explicit-any */
import axios, {
  AxiosInstance,
  AxiosRequestConfig,
  CreateAxiosDefaults,
} from "axios";
import { LaravelResponseObjectPromise } from "@/types/main/types";
import {
  RegisterUserParams,
  UpdateUserParams,
} from "../redux/reducers/user/userActions.d";
import {
  GetPropertyParams,
  GetPropertiesFeaturedParams,
  GetPropertiesSuperFeaturedParams,
  GetPropertiesParams,
  CreatePropertyParams,
  GenerateWithAiParams,
  AddKeyParams,
  EasyBrokersParams,
  DeleteKeyParams,
  UpdateKeyParams,
  EasyBrokersPropertiesParams,
  GetPropertyFromEasyBrokerParams,
  UpdatePropertyFromEasyBrokerParams,
  CreatePropertyFromEasyBrokerParams,
  ImportPropertyParams,
  getPropertyCompleteParams,
  HandleExcelsParams,
  getSeoIndexParams,
} from "../redux/reducers/properties/propertiesActions.d";
import { getFileToSend } from "@/lib/utils/get-file-to-send";
import Cookies from "js-cookie";

interface HttpCallableProps extends AxiosRequestConfig {
  noContentType?: boolean;
  headers?: Record<string, string>;
  method: "GET" | "POST" | "DELETE" | "PUT";
}

class BackendFetching {
  baseApiUrl?: string;
  backendApiUrl: string;
  apiManager: AxiosInstance;

  constructor() {
    this.baseApiUrl =
      process.env.NEXT_PUBLIC_NODE_ENV === "development"
        ? process.env.NEXT_PUBLIC_APP_LOCAL_BACKEND_API
        : process.env.NEXT_PUBLIC_APP_PROD_BACKEND_API;

    this.backendApiUrl = this.baseApiUrl + "/api";

    const axiosConfigs: CreateAxiosDefaults = {
      baseURL: this.backendApiUrl,
      responseType: "json",
      withCredentials: true,
    };

    this.apiManager = axios.create(axiosConfigs);
  }

  async httpCallable<T>(
    url: string,
    configs: HttpCallableProps
  ): LaravelResponseObjectPromise {
    const urlComplete = this.backendApiUrl + url;
    const config = {
      ...configs,
    };

    try {
      const response = await this.apiManager(urlComplete, config);

      return {
        error: false,
        data: response as T,
      };
    } catch (error: any) {
      if (axios.isAxiosError(error)) {
        return {
          error: true,
          message:
            error.response?.data?.message || "An unknown axios error occurred",
        };
      }
      return {
        error: true,
        message: error.message || "An unknown error occurred",
      };
    }
  }

  async httpAuthenticatedCallable<T>(
    url: string,
    configs: HttpCallableProps
  ): LaravelResponseObjectPromise {
    const accessToken = Cookies.get("access_token");
    const urlComplete = this.backendApiUrl + url;
    const config = {
      ...configs,
      headers: {
        ...configs.headers,
        Authorization: "Bearer " + accessToken,
      },
    };

    try {
      const response = await this.apiManager(urlComplete, config);

      return {
        error: false,
        data: response as T,
      };
    } catch (error: any) {
      if (axios.isAxiosError(error)) {
        return {
          error: true,
          message:
            error.response?.data?.message || "An unknown axios error occurred",
        };
      }
      return {
        error: true,
        message: error.message || "An unknown error occurred",
      };
    }
  }

  /**
   * general endpoints
   */
  async getCountries(): LaravelResponseObjectPromise {
    const response = await this.httpCallable("/countries", {
      method: "GET",
    });

    return response;
  }
  async getCountryPhoneCodes(): LaravelResponseObjectPromise {
    const response = await this.httpCallable("/countries/phone-codes", {
      method: "GET",
    });

    return response;
  }
  async getStates(country: string): LaravelResponseObjectPromise {
    const response = await this.httpCallable("/states/" + country, {
      method: "GET",
    });
    return response;
  }
  async getCities(state: string): LaravelResponseObjectPromise {
    const response = await this.httpCallable("/cities/" + state, {
      method: "GET",
    });
    return response;
  }
  async getUsersAdvisors(): LaravelResponseObjectPromise {
    const response = await this.httpAuthenticatedCallable(
      "/users/type/demonstrations?label=specialist",
      {
        method: "GET",
      }
    );
    return response;
  }
  async getUsersCommercialAdvisors(): LaravelResponseObjectPromise {
    const response = await this.httpAuthenticatedCallable(
      "/users/type/demonstrations?label=advisor",
      {
        method: "GET",
      }
    );
    return response;
  }

  async getColonies(
    country: number,
    city: number
  ): LaravelResponseObjectPromise {
    const response = await this.httpCallable(
      "/colonies/" + country + "/" + city,
      {
        method: "GET",
      }
    );
    return response;
  }
  /**
   * end of general endpoints
   */

  /**
   * Authentication endpoints
   */

  async loginWithEmail(
    email: string,
    password: string
  ): LaravelResponseObjectPromise {
    const response = await this.httpCallable("/login", {
      method: "POST",
      data: {
        email,
        password,
      },
    });

    return response;
  }

  async recoveryToken(): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable("/recoveryToken", {
      method: "GET",
    });

    return response;
  }

  async registerUser({
    name,
    lastName,
    email,
    password,
    confirmPassword,
    termsAndConditions,
  }: RegisterUserParams): LaravelResponseObjectPromise {
    const response = this.httpCallable("/register", {
      method: "POST",
      data: {
        name,
        lastName,
        email,
        password,
        confirmPassword,
        termsAndConditions: termsAndConditions ? 1 : 0,
      },
    });

    return response;
  }

  async updateUser({
    name,
    lastName,
    phone,
    officePhone,
    sex,
    profileCompany,
    showModalPassword,
    password,
    confirmPassword,
    companyName,
    country,
    state,
    city,
    companyLat,
    companyLng,
    companyLocation,
    twitter,
    youtube,
    linkedin,
    tiktok,
    instagram,
    facebookpage,
  }: UpdateUserParams): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable("/users/profile", {
      method: "POST",
      data: {
        name,
        lastName,
        phone,
        officePhone,
        sex,
        profileCompany,
        showModalPassword,
        password,
        confirmPassword,
        companyName,
        country,
        state,
        city,
        companyLat,
        companyLng,
        companyLocation,
        twitter,
        youtube,
        linkedin,
        tiktok,
        instagram,
        facebookpage,
      },
    });

    return response;
  }

  async getProfile(id: string): LaravelResponseObjectPromise {
    const response = this.httpCallable("/users/profile/" + id, {
      method: "GET",
    });

    return response;
  }

  /**
   * AuthenticationEndpoints end
   */

  /**
   * properties
   */

  async getProperty({ hash }: GetPropertyParams): LaravelResponseObjectPromise {
    const response = this.httpCallable("/properties/url/" + hash, {
      method: "GET",
    });

    return response;
  }

  async getPropertySuperFeatured({
    user_id,
  }: GetPropertiesSuperFeaturedParams): LaravelResponseObjectPromise {
    const response = this.httpCallable(
      "/featureds/properties/super?user_id=" + user_id,
      {
        method: "GET",
      }
    );

    return response;
  }

  async getPropertyFeatured({
    user_id,
  }: GetPropertiesFeaturedParams): LaravelResponseObjectPromise {
    const response = this.httpCallable(
      "/featureds/properties/simple?user_id=" + user_id,
      {
        method: "GET",
      }
    );

    return response;
  }

  async getProperties({
    user_id,
    offset,
    limit,
    type_id,
    rooms,
    bathrooms,
    parking,
    min_price,
    max_price,
    min_land_area,
    max_land_area,
    max_built_area,
    min_built_area,
    set_id,
    category_id,
    search,
    orientation,
  }: GetPropertiesParams): LaravelResponseObjectPromise {
    const response = this.httpCallable(
      `/properties/home?user_id=?${user_id}&offset=${offset}&limit=${limit}&type_id=${type_id}&rooms=${rooms}&bathrooms=${bathrooms}&parking=${parking}&min_price=${min_price}&max_price=${max_price}&min_land_area=${min_land_area}&max_land_area=${max_land_area}&max_built_area=${max_built_area}&min_built_area=${min_built_area}&set_id=${set_id}&category_id=${category_id}&search=${search}&orientation=${orientation}`,
      {
        method: "GET",
      }
    );

    return response;
  }

  async getPropertiesSearch({
    user_id,
    offset,
    limit,
    type_id,
    rooms,
    bathrooms,
    parking,
    min_price,
    max_price,
    min_land_area,
    max_land_area,
    max_built_area,
    min_built_area,
    set_id,
    category_id,
    search,
    orientation,
    city_id,
    state_id,
    colony_id,
  }: GetPropertiesParams): LaravelResponseObjectPromise {
    const response = this.httpCallable(`/properties/home`, {
      method: "GET",
      params: {
        user_id: user_id ?? "",
        offset: (offset ?? 0) * (limit ?? 10),
        limit: limit ?? 10,
        type_id: type_id ?? "",
        rooms: rooms ?? "",
        bathrooms: bathrooms ?? "",
        parking: parking ?? "",
        min_price: min_price ?? 0,
        max_price: max_price ?? 0,
        min_land_area: min_land_area ?? 0,
        max_land_area: max_land_area ?? 0,
        max_built_area: max_built_area ?? 0,
        min_built_area: min_built_area ?? 0,
        set_id: set_id ?? "",
        category_id: category_id ?? "",
        search: search ?? "",
        orientation: orientation ?? "",
        city_id: city_id ?? "",
        state_id: state_id ?? "",
        colony_id: colony_id ?? "",
      },
    });

    return response;
  }

  async getPropertiesUser({
    offset,
    limit,
  }: GetPropertiesParams): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(`/properties`, {
      method: "GET",
      params: {
        offset: (offset ?? 0) * (limit ?? 10),
        limit: limit ?? 10,
      },
    });

    return response;
  }

  async getPropertyTypes(): LaravelResponseObjectPromise {
    const response = this.httpCallable(`/types`, {
      method: "GET",
    });

    return response;
  }
  async getPropertyCategories(): LaravelResponseObjectPromise {
    const response = this.httpCallable(`/categorias`, {
      method: "GET",
    });

    return response;
  }
  async getPropertyUrbTypes(): LaravelResponseObjectPromise {
    const response = this.httpCallable(`/sets`, {
      method: "GET",
    });

    return response;
  }
  async createProperty({
    title,
    categoryId,
    price,
    isVideo,
    lat,
    lng,
    typeId,
    description,
    isAa,
    isWater,
    isFurnished,
    baths,
    isBeds,
    isCloset,
    isKitchen,
    isDiningRoom,
    isDisabled,
    isSchool,
    isStove,
    isGas,
    rooms,
    isOven,
    isWifi,
    isWashingMachine,
    isLight,
    m2Build,
    m2Property,
    isMaintenance,
    halfbaths,
    isMicrowave,
    others,
    isMiniOven,
    parking,
    isSwimmingPool,
    level,
    isRefrigerator,
    isLivingRoom,
    isDryingMachine,
    isVideoSurveillance,
    isPrivateSecurity,
    countryId,
    stateId,
    cityId,
    setId,
    antiquity,
    alias,
    images,
    videos,
  }: CreatePropertyParams): LaravelResponseObjectPromise {
    const formDataKeys = {
      title,
      categoryId,
      price,
      isVideo,
      lat,
      lng,
      typeId,
      description,
      isAa,
      isWater,
      isFurnished,
      baths,
      isBeds,
      isCloset,
      isKitchen,
      others,
      isDiningRoom,
      isDisabled,
      isSchool,
      isStove,
      isGas,
      rooms,
      isOven,
      isWifi,
      isWashingMachine,
      isLight,
      m2Build,
      m2Property,
      isMaintenance,
      halfbaths,
      isMicrowave,
      isMiniOven,
      parking,
      isSwimmingPool,
      level,
      isRefrigerator,
      isLivingRoom,
      isDryingMachine,
      isVideoSurveillance,
      isPrivateSecurity,
      countryId,
      stateId,
      cityId,
      setId,
      antiquity,
      alias,
    };

    const formData = new FormData();

    if (images) {
      images.forEach(async (d) => {
        formData.append("images", await getFileToSend(d.url));
      });
    }
    if (videos) {
      videos.forEach(async (d) => {
        formData.append("videos", await getFileToSend(d.url));
      });
    }

    for (const key in formDataKeys) {
      const value = formDataKeys[key as keyof typeof formDataKeys];
      if (value !== undefined && value !== null && key !== "file") {
        formData.append(key, String(value));
      }
    }

    const response = this.httpAuthenticatedCallable(`/properties`, {
      method: "POST",
      data: formData,
      headers: {
        "Content-Type": "multipart/form-data",
        Accept: "application/json",
      },
    });

    return response;
  }

  async generateWithAi(
    params: Omit<GenerateWithAiParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/openIA/properties/description`,
      {
        method: "GET",
        params,
      }
    );

    return response;
  }

  async easybrokersIndex({
    offset,
    limit,
  }: Omit<EasyBrokersParams, "context">): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(`/easybrokers/user`, {
      method: "GET",
      params: {
        offset: (offset ?? 0) * (limit ?? 10),
        limit: limit ?? 10,
      },
    });

    return response;
  }

  async easybrokersIndexProperties({
    page,
    id,
  }: Omit<
    EasyBrokersPropertiesParams,
    "context"
  >): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/easybrokers/properties/${id}`,
      {
        method: "GET",
        params: {
          page,
        },
      }
    );

    return response;
  }
  async easybrokersIndexPropertiesAllTags({
    id,
  }: Omit<
    EasyBrokersPropertiesParams,
    "context"
  >): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/easybrokers/properties/tags/${id}`,
      {
        method: "GET",
      }
    );

    return response;
  }

  async addKey(
    params: Omit<AddKeyParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(`/easybrokers`, {
      method: "POST",
      data: params,
    });

    return response;
  }
  async updateKey(
    params: Omit<UpdateKeyParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/easybrokers/${params?.id}`,
      {
        method: "POST",
        data: {
          name: params?.name,
          key: params?.key,
        },
      }
    );

    return response;
  }
  async deleteKey(
    params: Omit<DeleteKeyParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/easybrokers/${params?.id}`,
      {
        method: "DELETE",
      }
    );

    return response;
  }
  async getPropertyFromEasyBroker(
    params: Omit<GetPropertyFromEasyBrokerParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/easybrokers/show/${params?.id}/${params?.public_id}`,
      {
        method: "GET",
      }
    );

    return response;
  }

  async createPropertyFromEasyBroker(
    params: Omit<CreatePropertyFromEasyBrokerParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(`/propertiesbykeys`, {
      method: "POST",
      data: params,
    });

    return response;
  }
  async updatePropertyFromEasyBroker(
    params: Omit<UpdatePropertyFromEasyBrokerParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/propertiesbykeys/${params?.apiKey}`,
      {
        method: "POST",
        data: params,
      }
    );

    return response;
  }
  async importProperty(
    params: Omit<ImportPropertyParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/propertiesbykeys/import/data`,
      {
        method: "POST",
        data: params,
      }
    );

    return response;
  }
  async getPropertyComplete(
    params: Omit<getPropertyCompleteParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(`/propertiesbykeys`, {
      method: "GET",
      params,
    });

    return response;
  }

  async handleExcels(
    params: Omit<HandleExcelsParams, "context">
  ): LaravelResponseObjectPromise {
    const formData = new FormData();

    formData.append("excel", await getFileToSend(params?.excel.uri));

    if (params?.zip)
      formData.append("zip", await getFileToSend(params?.zip.uri));

    const response = this.httpAuthenticatedCallable(`/excels`, {
      method: "POST",
      data: formData,
      headers: {
        "Content-Type": "multipart/form-data",
        Accept: "application/json",
      },
    });

    return response;
  }

  async getSeoIndex(
    params: Omit<getSeoIndexParams, "context">
  ): LaravelResponseObjectPromise {
    const response = this.httpAuthenticatedCallable(
      `/seo/${params?.variation}?limit=${params?.limit}`,
      {
        method: "GET",
        params,
      }
    );

    return response;
  }

  /**
   * properties end
   */
}

export default BackendFetching;
