import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import {
  GetCountriesResponse,
  GetCountriesParams,
  GetStatesResponse,
  GetStatesParams,
  GetCitiesResponse,
  GetCitiesParams,
  GetCountryPhoneCodesResponse,
  GetCountryPhoneCodesParams,
} from "./appActions.d";
import { UserInterface } from "@/types/users/types";

interface initialStateProps {
  getCountries: {
    loadingCountries: boolean;
    countries: GetCountriesResponse["countries"];
    successCountries: boolean;
    errorCountries: string | null;
  };
  getCountryPhoneCodes: {
    loadingCountryPhoneCodes: boolean;
    countryPhoneCodes: GetCountryPhoneCodesResponse["countries"] | [];
    successCountryPhoneCodes: boolean;
    errorCountryPhoneCodes: string | null;
  };
  usersAdvisors: UserInterface[];
}

const initialState: initialStateProps = {
  getCountries: {
    loadingCountries: false,
    countries: [],
    successCountries: false,
    errorCountries: null,
  },
  getCountryPhoneCodes: {
    loadingCountryPhoneCodes: false,
    countryPhoneCodes: [],
    successCountryPhoneCodes: false,
    errorCountryPhoneCodes: null,
  },
  usersAdvisors: [],
};

export const getCountries = createAsyncThunk<
  GetCountriesResponse,
  GetCountriesParams
>("app/getCountries", async (params, { rejectWithValue }) => {
  try {
    const query = await params.context.getCountries();

    if (query.error) {
      throw query.message;
    }

    const response = query?.data?.data;

    return response.data;
  } catch (error) {
    return rejectWithValue(error);
  }
});

export const getCountryPhoneCodes = createAsyncThunk<
  GetCountryPhoneCodesResponse,
  GetCountryPhoneCodesParams
>("app/getCountryPhoneCodes", async (params, { rejectWithValue }) => {
  try {
    const query = await params.context.getCountryPhoneCodes();

    if (query.error) {
      throw query.message;
    }

    const response = query?.data?.data;

    return response.data;
  } catch (error) {
    return rejectWithValue(error);
  }
});

export const getStates = createAsyncThunk<GetStatesResponse, GetStatesParams>(
  "app/getStates",
  async (params, { rejectWithValue }) => {
    try {
      const query = await params.context.getStates(params.country);

      if (query.error) {
        throw query.message;
      }

      const response = query?.data?.data;
      return response;
    } catch (error) {
      return rejectWithValue(error);
    }
  }
);

export const getCities = createAsyncThunk<GetCitiesResponse, GetCitiesParams>(
  "app/getCities",
  async (params, { rejectWithValue }) => {
    try {
      const query = await params.context.getCities(params.state);

      if (query.error) {
        throw query.message;
      }

      const response = query?.data?.data;
      return response;
    } catch (error) {
      return rejectWithValue(error);
    }
  }
);

const appSlice = createSlice({
  name: "app",
  initialState,
  reducers: {},
  extraReducers: (builder) => {
    builder.addCase(getCountries.pending, (state) => {
      state.getCountries.loadingCountries = true;
    });
    builder.addCase(getCountries.fulfilled, (state, action) => {
      state.getCountries.loadingCountries = false;
      state.getCountries.successCountries = true;
      state.getCountries.countries = action.payload.countries;
    });
    builder.addCase(getCountries.rejected, (state, action) => {
      state.getCountries.loadingCountries = false;
      state.getCountries.errorCountries = action.payload as string;
    });

    builder.addCase(getCountryPhoneCodes.pending, (state) => {
      state.getCountryPhoneCodes.loadingCountryPhoneCodes = true;
    });
    builder.addCase(getCountryPhoneCodes.fulfilled, (state, action) => {
      state.getCountryPhoneCodes.loadingCountryPhoneCodes = false;
      state.getCountryPhoneCodes.successCountryPhoneCodes = true;
      state.getCountryPhoneCodes.countryPhoneCodes = action.payload.countries;
    });
    builder.addCase(getCountryPhoneCodes.rejected, (state, action) => {
      state.getCountryPhoneCodes.loadingCountryPhoneCodes = false;
      state.getCountryPhoneCodes.errorCountryPhoneCodes =
        action.payload as string;
    });
  },
});

export default appSlice.reducer;
