import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";
import {
  GetProfileParams,
  GetProfileResponse,
  loginWithEmailParams,
  loginWithEmailResponse,
  recoveryTokenParams,
  recoveryTokenResponse,
  RegisterUserParams,
  RegisterUserResponse,
  UpdateUserParams,
  UpdateUserResponse,
} from "./userActions.d";
import { UserInterface } from "@/types/users/types";
import Cookies from "js-cookie";

interface initialStateProps {
  userData: UserInterface | null;
  hideRegisterFlow: boolean;
  showOnlyRegisterPage: boolean;
}

const initialState: initialStateProps = {
  userData: null,
  hideRegisterFlow: false,
  showOnlyRegisterPage: false,
};

export const loginWithEmail = createAsyncThunk<
  loginWithEmailResponse,
  loginWithEmailParams
>("user/loginWithEmail", async (params, { rejectWithValue }) => {
  try {
    const query = await params.context.loginWithEmail(
      params.email,
      params.password
    );

    if (query.error) {
      throw new Error(query.message);
    }

    if (query.data.data.error) {
      throw new Error(query.data.data.message);
    }

    const response: loginWithEmailResponse = query.data.data.data;

    Cookies.set("access_token", response?.access_token, {
      expires: 7,
      secure: process.env.NODE_ENV === "production",
      sameSite: "Strict",
      path: "/",
    });
    Cookies.set("user_cached_data", JSON.stringify(response?.users), {
      expires: 7,
      secure: process.env.NODE_ENV === "production",
      sameSite: "Strict",
      path: "/",
    });

    return response;
  } catch (error) {
    return rejectWithValue(error);
  }
});

export const recoveryToken = createAsyncThunk<
  recoveryTokenResponse,
  recoveryTokenParams
>(
  "users/recoveryToken",
  async (
    params,
    { rejectWithValue, dispatch }
  ): Promise<recoveryTokenResponse | ReturnType<typeof rejectWithValue>> => {
    try {
      const user_cached_data = Cookies.get("user_cached_data");

      if (user_cached_data) {
        dispatch(setUserData(JSON.parse(user_cached_data)));
      }

      const token = Cookies.get("access_token");

      if (token) {
        const query = await params.context.recoveryToken();

        if (query.error) {
          throw new Error(query.message);
        }

        const response: recoveryTokenResponse = query.data.data.data;

        Cookies.set("access_token", response?.access_token, {
          expires: 7,
          secure: process.env.NODE_ENV === "production",
          sameSite: "Strict",
          path: "/",
        });
        Cookies.set("user_cached_data", JSON.stringify(response?.user), {
          expires: 7,
          secure: process.env.NODE_ENV === "production",
          sameSite: "Strict",
          path: "/",
        });

        return response;
      }

      return {
        access_token: "",
        user: user_cached_data ? JSON.parse(user_cached_data) : null,
      };
    } catch (error) {
      console.log("error", error);
      return rejectWithValue(error);
    }
  }
);

export const registerUser = createAsyncThunk<
  RegisterUserResponse,
  RegisterUserParams
>("user/registerUser", async (params, { rejectWithValue }) => {
  try {
    const query = await params.context.registerUser(params);

    if (query.error) {
      throw new Error(query.message);
    }

    if (query.data.data.error) {
      throw new Error(query.data.data.message);
    }

    const response: RegisterUserResponse = query.data.data.data;

    return response;
  } catch (error) {
    return rejectWithValue(error);
  }
});

export const updateUser = createAsyncThunk<
  UpdateUserResponse,
  UpdateUserParams
>("user/updateUser", async (params, { rejectWithValue }) => {
  try {
    const query = await params.context.updateUser(params);

    if (query.error) {
      throw new Error(query.message);
    }

    if (query.data.data.error) {
      throw new Error(query.data.data.message);
    }

    const response: UpdateUserResponse = query.data.data.data;

    return response;
  } catch (error) {
    return rejectWithValue(error);
  }
});

export const getProfile = createAsyncThunk<
  GetProfileResponse,
  GetProfileParams
>("user/getProfile", async (params, { rejectWithValue }) => {
  try {
    const query = await params.context.getProfile(params?.id);

    if (query.error) {
      throw new Error(query.message);
    }

    if (query.data.data.error) {
      throw new Error(query.data.data.message);
    }

    const response: GetProfileResponse = query.data.data.data;

    return response;
  } catch (error) {
    return rejectWithValue(error);
  }
});

const userSlice = createSlice({
  name: "user",
  initialState,
  reducers: {
    setUserData: (state, action) => {
      state.userData = action.payload;
    },
    userLogout: (state) => {
      Cookies.remove("access_token");
      Cookies.remove("user_cached_data");

      state.userData = null;
    },
    changeHideRegisterFlow: (state, action) => {
      state.hideRegisterFlow = action.payload ?? false;
    },
    changeShowOnlyRegisterFlow: (state, action) => {
      state.showOnlyRegisterPage = action.payload ?? false;
    },
  },
  extraReducers: (builder) => {
    builder.addCase(loginWithEmail.fulfilled, (state, action) => {
      state.userData = action.payload.users;
    });

    builder.addCase(recoveryToken.fulfilled, (state, action) => {
      state.userData = action.payload.user;
    });
  },
});

export const {
  userLogout,
  setUserData,
  changeShowOnlyRegisterFlow,
  changeHideRegisterFlow,
} = userSlice.actions;

export default userSlice.reducer;
