/* eslint-disable react-hooks/exhaustive-deps */
import React, { useCallback, useEffect, useMemo, useState } from "react";
import styles from "./styles.module.scss";
import PropertyCard from "@/components/PropertyCard/PropertyCard";
import {
  Button,
  CircularProgress,
  MenuItem,
  Pagination,
  Select,
  useMediaQuery,
} from "@mui/material";
import { useTranslations } from "next-intl";
import { TuneOutlined, WarningRounded } from "@mui/icons-material";
import PersonProfileCard from "@/components/PersonProfileCard/PersonProfileCard";
import GoogleMapComponent from "@/components/GoogleMapComponent/GoogleMapComponent";
import FiltersModal from "./components/FiltersModal/FiltersModal";
import { SearchObject } from "../Results/components/MoreFilters/moreFilters";
import {
  GetPropertiesParams,
  GetPropertiesResponse,
  IPropertyFeatured,
} from "@/lib/services/redux/reducers/properties/propertiesActions.d";
import { getPropertiesSearch } from "@/lib/services/redux/reducers/properties/propertiesActions";
import useRequestHandler from "@/hooks/useRequestHandler/useRequestHandler";
import { calculateTotalPages } from "@/lib/utils/paginationUtils";
import { GetProfileResponse } from "@/lib/services/redux/reducers/user/userActions.d";
import { getProfile } from "@/lib/services/redux/reducers/user/userActions";

interface ProfileViewProps {
  id?: string;
}

const Profile: React.FC<ProfileViewProps> = ({ id }) => {
  const t = useTranslations();

  const [openModal, setOpenModal] = useState(false);
  const [profileData, setProfileData] = useState<
    GetProfileResponse["users"] | null
  >(null);

  const { fetcher: getPropertiesFetch, requestStatus: propertiesStatus } =
    useRequestHandler<GetPropertiesResponse, GetPropertiesParams>(
      getPropertiesSearch
    );
  const { fetcher: getProfileFetch } = useRequestHandler<
    GetProfileResponse,
    GetPropertiesParams
  >(getProfile);

  const [properties, setProperties] = useState<IPropertyFeatured[]>([]);
  const [totalHere, setTotalHere] = useState<number>(0);

  const is693 = useMediaQuery("(max-width: 693px)");
  const is571 = useMediaQuery("(max-width: 571px)");

  const cardsMinWidth = useMemo(() => {
    if (is693 && !is571) {
      return "47.5%";
    }
    if (is571) {
      return "100%";
    }

    return "304px";
  }, [is693]);

  const [filters, setFilters] = useState<SearchObject>({
    offset: 0,
    limit: 12,
  });

  const retryFunction = useCallback(
    async (currentFilters?: Partial<SearchObject>) => {
      const payload = {
        offset: currentFilters?.offset ?? filters?.offset ?? 0,
        limit: currentFilters?.limit ?? filters?.limit ?? 12,
        user_id: id,
      };

      const query = await getPropertiesFetch(payload);

      if (query.status === "success") {
        setProperties(query.response.properties);
        setTotalHere(query.response.total ?? 0);
      }

      const profilequery = await getProfileFetch({
        id,
      });

      if (profilequery.status === "success") {
        setProfileData(profilequery.response.users);
      }
    },
    [filters, setProperties, setTotalHere]
  );

  useEffect(() => {
    retryFunction();
  }, [id]);

  return (
    <div className={styles.container}>
      <div className={styles.profileContainer}>
        <PersonProfileCard profile={profileData} />
        <div className={styles.mapPart}>
          {Boolean(profileData?.lat && profileData?.lng) && (
            <GoogleMapComponent
              center={{
                lat: profileData?.lat ?? 0,
                lng: profileData?.lng ?? 0,
              }}
              onChangePosition={function ({}: { lat: number; lng: number }) {
                // throw new Error("Function not implemented.");
              }}
            />
          )}
          {Boolean(!profileData?.lat || !profileData?.lng) && (
            <div className={styles.shadow}>
              <p>{t("mapNotAvailable")}</p>
            </div>
          )}
        </div>
      </div>
      <div className={styles.properties}>
        <div className={styles.filters}>
          <Button
            variant="contained"
            disableElevation
            className={styles.addProperty}
            startIcon={<TuneOutlined htmlColor={"#13A407"} />}
            onClick={() => setOpenModal(true)}
          >
            {t("filters")}
          </Button>
        </div>

        <div className={styles.propertiesRowWrap}>
          {properties.length === 0 && !propertiesStatus.loading && (
            <div className={styles.noData}>
              <WarningRounded sx={{ fontSize: "100px" }} htmlColor="#DDDDDD" />
              <p>{t("noPropertiesFound")}</p>
            </div>
          )}
          {propertiesStatus.loading && (
            <div className={styles.noData}>
              <CircularProgress size={"50px"} color="primary" />
            </div>
          )}
          {properties.map((d) => {
            return (
              <PropertyCard
                fulldata={d}
                key={d.id}
                type={
                  d.isSuperFeatured
                    ? "super"
                    : d.isFeatured
                    ? "simple"
                    : "original"
                }
                title={d.titulo}
                totalBeds={d?.habitaciones ?? 0}
                totalBaths={d?.baños ?? 0}
                totalHomes={Number(d?.pisos) ?? 1}
                totalParkings={d?.parking ?? 0}
                price={d.precio}
                user={d.users}
                categoy={
                  d.categories
                    ? d.categories.nombre === "Venta"
                      ? "sale"
                      : "rent"
                    : "rent"
                }
                description={d.descripcion}
                img={d.imgs && d.imgs.length > 0 ? d.imgs[0] : null}
                userImg={d.users?.img ?? null}
                userTotalProperties={d.users?.totalProperties ?? 0}
                userId={d.users?.id ?? ""}
                slug={d.slug}
                minWidth={cardsMinWidth}
              />
            );
          })}
        </div>

        {calculateTotalPages(totalHere ?? 0, 12) > 0 && (
          <div className={styles.paginationContainer}>
            <Pagination
              count={calculateTotalPages(totalHere ?? 0, 12)}
              color="primary"
              page={(filters?.offset ?? 0) + 1}
              onChange={(_, page) => {
                if (setFilters) {
                  const payload = {
                    ...filters,
                    offset: page - 1,
                  };
                  setFilters(payload);
                  retryFunction(payload);
                }
              }}
            />
            <Select
              value={filters?.limit}
              size="small"
              onChange={(event) => {
                if (setFilters) {
                  const payload = {
                    ...filters,
                    limit: event.target.value - 1,
                  };
                  setFilters(payload);
                  retryFunction(payload);
                }
              }}
            >
              {[12, 24, 32, 64, 128, 256, 512, 1024].map((p) => {
                return (
                  <MenuItem key={p} value={p}>
                    {p}
                  </MenuItem>
                );
              })}
            </Select>
          </div>
        )}
        <FiltersModal
          modalProps={{ isOpen: openModal, toggle: () => setOpenModal(false) }}
          filters={filters}
          setFilters={setFilters}
          retrySearchFunction={retryFunction}
        />
      </div>
    </div>
  );
};

export default Profile;
