/* eslint-disable react-hooks/exhaustive-deps */
import React, { useCallback, useEffect, useState } from "react";
import styles from "./styles.module.scss";
import { useTranslations } from "next-intl";
import { Button, Fade, IconButton, InputBase, Popper } from "@mui/material";
import { TuneOutlined } from "@mui/icons-material";
import { PRIMARY_COLOR, WHITE } from "@/styles/constants";
import {
  GetPropertUrbTypesParams,
  GetPropertUrbTypesResponse,
} from "@/lib/services/redux/reducers/properties/propertiesActions.d";
import { GetPropertyUrbTypes } from "@/lib/services/redux/reducers/properties/propertiesActions";
import useRequestHandler from "@/hooks/useRequestHandler/useRequestHandler";
import { SelectOptionsProps } from "@/components/FormInput/types";
import useStyles from "./sxStyles";

export interface SearchObject {
  search?: string;
  category?: string;
  city_id?: string;
  state_id?: string;
  colony_id?: string;
  propertyType?: string[];
  rooms?: number;
  bathRooms?: number;
  parkings?: number;
  price?: number[]; // this is a number range
  terrain?: number[]; // this is a number range
  built?: number[]; // this is a number range
  urbType?: string[];
  limit?: number;
  offset?: number;
  orientation?: "vertical" | "horizontal";
  ignore_search?: boolean
}

interface MoreFiltersProps {
  filters: SearchObject;
  setFilters: (currentFilters: Partial<SearchObject>) => unknown;
  openModal?: () => unknown;
}

const MoreFilters: React.FC<MoreFiltersProps> = ({
  filters,
  setFilters,
  openModal,
}) => {
  const t = useTranslations();

  const [anchorEl, setAnchorEl] = useState<HTMLElement | null>(null);
  const classes = useStyles();
  const open = Boolean(anchorEl);

  const { fetcher: fetcherUrbTypes } = useRequestHandler<
    GetPropertUrbTypesResponse,
    GetPropertUrbTypesParams
  >(GetPropertyUrbTypes);

  const [urbTypes, setUrbTypes] = useState<SelectOptionsProps[]>([]);

  const quantities = Array(6)
    .fill(0)
    .map((d, i) => {
      return {
        label: `${d + i + 1}+`,
        name: `${d + i + 1}+`,
        value: d + i + 1,
      };
    });

  const retryFunction = async () => {
    const urbTypeshere = await fetcherUrbTypes({});

    if (urbTypeshere.status === "success") {
      setUrbTypes(
        urbTypeshere.response.sets.map((d) => {
          return {
            label: d.nombre,
            name: d.id,
            value: d.id,
          };
        })
      );
    }
  };

  useEffect(() => {
    retryFunction();
  }, []);

  const handlePropertyUrbTypes = useCallback(
    (val: string) => {
      const values = [...(filters.urbType ?? [])];

      if (values.includes(val)) {
        setFilters({
          ...filters,
          urbType: values.filter((d) => d !== val),
        });
      } else {
        setFilters({ ...filters, urbType: values.concat(val) });
      }
    },
    [filters]
  );

  const handleClean = () => {
    setFilters({
      price: [0, 0],
      terrain: [0, 0],
      built: [0, 0],
    });
  };

  const handleSubmit = useCallback(() => {
    setAnchorEl(null);
  }, []);

  return (
    <>
      <div className={styles.buttonsoftop}>
        <Button
          // ref={mainContainerRef}
          variant="contained"
          disableElevation
          className={styles.moreFiltersbutton}
          startIcon={<TuneOutlined htmlColor={"#13A407"} />}
          onClick={(e) => {
            setAnchorEl(anchorEl ? null : (e.target as HTMLElement));
          }}
        >
          {t("moreFilters")}
        </Button>
        <IconButton className={styles.moreFiltersResp} onClick={openModal}>
          <TuneOutlined htmlColor={"#13A407"} />
        </IconButton>
      </div>
      <Popper
        open={open}
        anchorEl={anchorEl}
        placement={"bottom-end"}
        className={classes.popper}
        transition
      >
        {({ TransitionProps }) => (
          <Fade {...TransitionProps} timeout={350}>
            <div className={classes.container}>
              <div
                className={styles.rowOfCards}
                style={{ flexDirection: "column" }}
              >
                <div className={styles.cardLimit} style={{ width: "100%" }}>
                  <div className={styles.card}>
                    <p className={styles.title}>{t("bathRooms")}</p>
                    {quantities.map((d) => {
                      return (
                        <Button
                          variant="contained"
                          key={d.name}
                          disableElevation
                          className={styles.button}
                          style={{
                            background:
                              filters.bathRooms === d.value
                                ? PRIMARY_COLOR
                                : WHITE,
                            color:
                              filters.bathRooms !== d.value
                                ? PRIMARY_COLOR
                                : WHITE,
                          }}
                          onClick={() =>
                            setFilters({ ...filters, bathRooms: d.value })
                          }
                        >
                          {d.label}
                        </Button>
                      );
                    })}
                  </div>
                </div>
                <div className={styles.cardLimit} style={{ width: "100%" }}>
                  <div className={styles.card}>
                    <p className={styles.title}>{t("parkings")}</p>
                    {quantities.map((d) => {
                      return (
                        <Button
                          variant="contained"
                          key={d.name}
                          disableElevation
                          className={styles.button}
                          style={{
                            background:
                              filters.parkings === d.value
                                ? PRIMARY_COLOR
                                : WHITE,
                            color:
                              filters.parkings !== d.value
                                ? PRIMARY_COLOR
                                : WHITE,
                          }}
                          onClick={() =>
                            setFilters({ ...filters, parkings: d.value })
                          }
                        >
                          {d.label}
                        </Button>
                      );
                    })}
                  </div>
                </div>
              </div>
              <div className={styles.rowOfCards}>
                <div className={styles.cardLimit}>
                  <div
                    className={styles.card}
                    style={{ height: "fit-content" }}
                  >
                    <p className={styles.title}>{t("terrain")}</p>
                    <div className={styles.inputRow}>
                      <InputBase
                        className={styles.input}
                        placeholder={t("from")}
                        type="number"
                        value={filters?.terrain?.[0] ?? undefined}
                        onChange={(d) => {
                          const value = [...(filters.terrain ?? [])];
                          value[0] = Number(d.target.value);

                          setFilters({ ...filters, terrain: value });
                        }}
                      />
                      <InputBase
                        className={styles.input}
                        placeholder={t("until")}
                        type="number"
                        value={filters?.terrain?.[1] ?? undefined}
                        onChange={(d) => {
                          const value = [...(filters.terrain ?? [])];
                          value[1] = Number(d.target.value);

                          setFilters({ ...filters, terrain: value });
                        }}
                      />
                    </div>
                  </div>
                </div>
                <div className={styles.cardLimit}>
                  <div
                    className={styles.card}
                    style={{ height: "fit-content" }}
                  >
                    <p className={styles.title}>{t("built")}</p>
                    <div className={styles.inputRow}>
                      <InputBase
                        className={styles.input}
                        placeholder={t("from")}
                        type="number"
                        value={filters?.built?.[0] ?? undefined}
                        onChange={(d) => {
                          const value = [...(filters.built ?? [])];
                          value[0] = Number(d.target.value);

                          setFilters({ ...filters, built: value });
                        }}
                      />
                      <InputBase
                        className={styles.input}
                        placeholder={t("until")}
                        type="number"
                        value={filters?.built?.[1] ?? undefined}
                        onChange={(d) => {
                          const value = [...(filters.built ?? [])];
                          value[1] = Number(d.target.value);

                          setFilters({ ...filters, built: value });
                        }}
                      />
                    </div>
                  </div>
                </div>
              </div>
              <div className={styles.card}>
                <p className={styles.title}>{t("urbType")}</p>
                {urbTypes.map((d) => {
                  return (
                    <Button
                      variant="contained"
                      key={d.value}
                      disableElevation
                      className={styles.button}
                      style={{
                        minWidth: "47.5%",
                        background: filters.urbType?.includes(d.value)
                          ? PRIMARY_COLOR
                          : WHITE,
                        color: !filters.urbType?.includes(d.value)
                          ? PRIMARY_COLOR
                          : WHITE,
                      }}
                      onClick={() => handlePropertyUrbTypes(d.value)}
                    >
                      {d.label}
                    </Button>
                  );
                })}
              </div>
              <div className={styles.divisor}></div>
              <div className={styles.buttons}>
                <Button
                  variant="contained"
                  disableElevation
                  className={styles.buttonStyle}
                  onClick={handleClean}
                >
                  {t("clean")}
                </Button>
                <Button
                  variant="contained"
                  disableElevation
                  className={styles.buttonStyle}
                  style={{ borderColor: PRIMARY_COLOR, color: PRIMARY_COLOR }}
                  onClick={handleSubmit}
                >
                  {t("viewResults")}
                </Button>
              </div>
            </div>
          </Fade>
        )}
      </Popper>
    </>
  );
};

export default MoreFilters;
