/* eslint-disable react-hooks/exhaustive-deps */
import React, { SetStateAction, useEffect, useMemo, useState } from "react";
import styles from "./styles.module.scss";
import FormInput from "@/components/FormInput/FormInput";
import { useTranslations } from "next-intl";
import {
  GetPropertCategoriesParams,
  GetPropertCategoriesResponse,
  GetPropertTypesParams,
  GetPropertTypesResponse,
} from "@/lib/services/redux/reducers/properties/propertiesActions.d";
import useRequestHandler from "@/hooks/useRequestHandler/useRequestHandler";
import {
  GetPropertyCategories,
  GetPropertyTypes,
} from "@/lib/services/redux/reducers/properties/propertiesActions";
import { SelectOptionsProps } from "@/components/FormInput/types";
import RangeSelector from "@/components/RangeSelector/RangeSelector";
import { formatCurrencyMXN } from "@/lib/utils/format-currenty-mxn";
import MoreFilters, { SearchObject } from "../MoreFilters/moreFilters";
import cardsIcon from "@/assets/icons/cardsIcon.svg";
import listIcon from "@/assets/icons/listIcon.svg";
import Image from "next/image";
import { CircularProgress } from "@mui/material";
import { debounce } from "lodash";

interface ResultsFilteringComponentProps {
  renderMode: "card" | "list";
  setRenderMode: React.Dispatch<SetStateAction<"card" | "list">>;
  filters: SearchObject;
  setFilters: React.Dispatch<SetStateAction<SearchObject>>;
  setignoreCurrentsearch: React.Dispatch<SetStateAction<boolean>>;
  defaultValues: SearchObject;
  retrySearchFunction: (currentFilters?: Partial<SearchObject>) => unknown;
  loading?: boolean;
  total: number;
  openModal?: () => unknown;
}

const ResultsFilteringComponent: React.FC<ResultsFilteringComponentProps> = ({
  renderMode,
  setRenderMode,
  filters,
  setFilters: setFiltersNoReal,
  setignoreCurrentsearch,
  loading,
  total,
  retrySearchFunction,
  defaultValues,
  openModal,
}) => {
  const t = useTranslations();

  const { fetcher: fetcherTypes } = useRequestHandler<
    GetPropertTypesResponse,
    GetPropertTypesParams
  >(GetPropertyTypes);
  const { fetcher: fetcherCategories } = useRequestHandler<
    GetPropertCategoriesResponse,
    GetPropertCategoriesParams
  >(GetPropertyCategories);

  const [propertyTypes, setPropertyTypes] = useState<SelectOptionsProps[]>([]);
  const [categories, setCategories] = useState<SelectOptionsProps[]>([]);

  const quantities = Array(6)
    .fill(0)
    .map((d, i) => {
      return {
        label: `${d + i + 1}+`,
        name: `${d + i + 1}+`,
        value: d + i + 1,
      };
    });

  const retryFunction = async () => {
    const typeshere = await fetcherTypes({});

    if (typeshere.status === "success") {
      setPropertyTypes(
        typeshere.response.types.map((d) => {
          return {
            label: d.nombre,
            name: d.id,
            value: d.id,
          };
        })
      );
    }

    const categorieshere = await fetcherCategories({});

    if (categorieshere.status === "success") {
      setCategories(
        categorieshere.response.categories.map((d) => {
          return {
            label: d.nombre,
            name: d.id,
            value: d.id,
          };
        })
      );
    }
  };

  useEffect(() => {
    retryFunction();
  }, []);

  const debouncedSearch = useMemo(
    () =>
      debounce((currentFilters: Partial<SearchObject>) => {
        retrySearchFunction(currentFilters);
        setignoreCurrentsearch(false);
      }, 1000),
    []
  );

  const setFilters = (currentFilters: Partial<SearchObject>) => {
    setFiltersNoReal(currentFilters);
    debouncedSearch(currentFilters);
  };

  return (
    <div className={styles.container}>
      <p className={styles.title}>{defaultValues.search}</p>
      <div className={styles.divider}></div>
      <div className={styles.filtersContainer}>
        <div className={styles.search}>
          <FormInput
            required={false}
            variationConfig={{
              type: "text",
              fullWidth: true,
              inputProps: {
                value: "",
                placeholder: "",
                style: {
                  width: "100%",
                },
                onChange: (e) => {
                  const payload = {
                    ...filters,
                    search: e.target.value,
                  };

                  setFiltersNoReal(payload);
                  debouncedSearch(payload);
                },
              },
            }}
          />
        </div>
        <div className={styles.selector}>
          <FormInput
            required={false}
            variationConfig={{
              type: "selector",
              fullWidth: true,
              inputProps: {
                options: categories,
                selectedItem: filters.category as string,
                placeholder: t("category"),
                onClickItem: (val) => {
                  setFilters({
                    ...filters,
                    category: val as string,
                  });
                },
              },
            }}
          />
        </div>
        <div className={styles.selector}>
          <FormInput
            required={false}
            variationConfig={{
              type: "selector",
              fullWidth: true,
              inputProps: {
                options: propertyTypes,
                selectedItem: (filters.propertyType?.[0] as string) ?? "",
                placeholder: t("propertyType"),
                onClickItem: (val) => {
                  setFilters({
                    ...filters,
                    propertyType: [val as string],
                  });
                },
              },
            }}
          />
        </div>
        <div className={styles.selector}>
          <FormInput
            required={false}
            variationConfig={{
              type: "selector",
              fullWidth: true,
              inputProps: {
                options: quantities,
                selectedItem: filters.rooms ?? null,
                placeholder: t("rooms"),
                onClickItem: (val) => {
                  setFilters({
                    ...filters,
                    rooms: val as number,
                  });
                },
              },
            }}
          />
        </div>
        <div className={styles.selector}>
          <RangeSelector
            placeholder={t(
              filters?.price
                ? filters?.price.map((d) => formatCurrencyMXN(d)).join(" - ")
                : "price"
            )}
            value={filters?.price ?? []}
            setValue={(e) => setFilters({ ...filters, price: e })}
          />
        </div>
        <MoreFilters
          filters={filters}
          setFilters={setFilters}
          openModal={openModal}
        />
      </div>

      <div className={styles.filtersContainer}>
        <p className={styles.totals}>{`${total ?? 0}  ${t("resultsOf")}  "${
          filters?.search ?? "--"
        }"`}</p>

        <div className={styles.orderAndRenderization}>
          {loading && (
            <CircularProgress size={"20px"} style={{ color: "#000000" }} />
          )}
          {/* <div className={styles.search}>
            <FormInput
              required={false}
              variationConfig={{
                type: "selector",
                fullWidth: true,
                inputProps: {
                  options: ["vertical", "horizontal"].map((d) => {
                    return {
                      label: t(d),
                      name: d,
                      value: d,
                    };
                  }),
                  selectedItem: filters.orientation as string,
                  placeholder: t("orderBy"),
                  onClickItem: (val) => {
                    setFilters({
                      ...filters,
                      orientation: val as "vertical" | "horizontal",
                    });
                  },
                },
              }}
            />
          </div> */}
          <div className={styles.renderizationcontrol}>
            <div
              className={styles.button}
              style={{
                borderRight: "1px solid #DDD",
                background: renderMode === "card" ? "#EDEDED" : "#FFF",
              }}
              onClick={() => {
                setRenderMode("card");
                setFilters({
                  ...filters,
                  orientation: "vertical",
                });
              }}
            >
              <Image src={cardsIcon} alt="cardsIcon" />
            </div>
            <div
              className={styles.button}
              style={{
                background: renderMode === "list" ? "#EDEDED" : "#FFF",
              }}
              onClick={() => {
                setRenderMode("list");
                setFilters({
                  ...filters,
                  orientation: "horizontal",
                });
              }}
            >
              <Image src={listIcon} alt="listIcon" />
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ResultsFilteringComponent;
