/* eslint-disable react-hooks/exhaustive-deps */
import React, { useCallback, useEffect, useMemo, useState } from "react";
import styles from "./styles.module.scss";
import ResultsFilteringComponent from "./components/ResultsFilteringComponent/ResultsFilteringComponent";
import { useSearchParams } from "next/navigation";
import { WarningRounded } from "@mui/icons-material";
import { useTranslations } from "next-intl";
import { SearchObject } from "./components/MoreFilters/moreFilters";
import useRequestHandler from "@/hooks/useRequestHandler/useRequestHandler";
import {
  GetPropertiesParams,
  GetPropertiesResponse,
  IPropertyFeatured,
} from "@/lib/services/redux/reducers/properties/propertiesActions.d";
import { getPropertiesSearch } from "@/lib/services/redux/reducers/properties/propertiesActions";
import PropertyCard from "@/components/PropertyCard/PropertyCard";
import { calculateTotalPages } from "@/lib/utils/paginationUtils";
import { MenuItem, Pagination, Select, useMediaQuery } from "@mui/material";
import FiltersModal from "../Profile/components/FiltersModal/FiltersModal";

const Results = () => {
  const t = useTranslations();
  const searchParams = useSearchParams();

  const [renderMode, setRenderMode] = useState<"card" | "list">("card");

  const { fetcher: getPropertiesFetch, requestStatus: propertiesStatus } =
    useRequestHandler<GetPropertiesResponse, GetPropertiesParams>(
      getPropertiesSearch
    );

  const [ignoreCurrentsearch, setignoreCurrentsearch] = useState(false);

  const [properties, setProperties] = useState<IPropertyFeatured[]>([]);
  const [openModal, setOpenModal] = useState<boolean>(false);
  const [totalHere, setTotalHere] = useState<number>(0);

  const is693 = useMediaQuery("(max-width: 693px)");
  const is571 = useMediaQuery("(max-width: 571px)");

  const cardsMinWidth = useMemo(() => {
    if (is693 && !is571) {
      return "47.5%";
    }
    if (is571) {
      return "100%";
    }

    return "304px";
  }, [is693]);

  const defaultValues = {
    search: searchParams.get("search") ?? undefined,
    category: searchParams.get("category") ?? undefined,
    propertyType: searchParams.get("propertyType")
      ? searchParams.get("propertyType")!.split(",")
      : undefined,
    rooms: Number(searchParams.get("rooms")) ?? undefined,
    bathRooms: Number(searchParams.get("bathRooms")) ?? undefined,
    parkings: Number(searchParams.get("parkings")) ?? undefined,
    price: searchParams.get("price")
      ? searchParams
          .get("price")!
          .split(",")
          .map((d) => Number(d))
      : undefined,
    terrain: searchParams.get("terrain")
      ? searchParams
          .get("terrain")!
          .split(",")
          .map((d) => Number(d))
      : undefined,
    built: searchParams.get("built")
      ? searchParams
          .get("built")!
          .split(",")
          .map((d) => Number(d))
      : undefined,
    urbType: searchParams.get("urbType")
      ? searchParams.get("urbType")!.split(",")
      : undefined,
    city_id: searchParams.get("city_id")
      ? searchParams.get("city_id")!
      : undefined,
    state_id: searchParams.get("state_id")
      ? searchParams.get("state_id")!
      : undefined,
    colony_id: searchParams.get("colony_id")
      ? searchParams.get("colony_id")!
      : undefined,
    ignore_search: searchParams.get("ignore_search")
      ? searchParams.get("ignore_search") === "true"
      : undefined,
  };

  useEffect(() => {
    if (defaultValues?.ignore_search) {
      setignoreCurrentsearch(true);
    }
  }, [defaultValues?.ignore_search]);

  const [filters, setFilters] = useState<SearchObject>(
    defaultValues
      ? {
          ...defaultValues,
          offset: 0,
          limit: 12,
          orientation: "vertical",
        }
      : {
          offset: 0,
          limit: 12,
          orientation: "vertical",
        }
  );

  const retryFunction = useCallback(
    async (currentFilters?: Partial<SearchObject>) => {
      const payload = {
        offset: currentFilters?.offset ?? filters?.offset ?? 0,
        limit: currentFilters?.limit ?? filters?.limit ?? 10,
        type_id:
          currentFilters?.propertyType?.[0] ?? filters?.propertyType?.[0],
        rooms: currentFilters?.rooms ?? filters?.rooms,
        bathrooms: currentFilters?.bathRooms ?? filters?.bathRooms,
        parking: currentFilters?.parkings ?? filters?.parkings,
        min_price: currentFilters?.price?.[0] ?? filters?.price?.[0],
        max_price: currentFilters?.price?.[1] ?? filters?.price?.[1],
        min_land_area: currentFilters?.terrain?.[0] ?? filters?.terrain?.[0],
        max_land_area: currentFilters?.terrain?.[1] ?? filters?.terrain?.[1],
        max_built_area: currentFilters?.built?.[0] ?? filters?.built?.[0],
        min_built_area: currentFilters?.built?.[1] ?? filters?.built?.[1],
        set_id: currentFilters?.urbType ?? filters?.urbType,
        category_id: currentFilters?.category ?? filters?.category,
        search: ignoreCurrentsearch
          ? ""
          : currentFilters?.search ?? filters?.search,
        orientation: currentFilters?.orientation ?? filters?.orientation,
        city_id: currentFilters?.city_id ?? filters?.city_id,
        state_id: currentFilters?.state_id ?? filters?.state_id,
        colony_id: currentFilters?.colony_id ?? filters?.colony_id,
      };

      const query = await getPropertiesFetch(payload);

      if (query.status === "success") {
        setProperties(query.response.properties);
        setTotalHere(query.response.total ?? 0);
      }
    },
    [filters, setProperties, setTotalHere, ignoreCurrentsearch]
  );

  useEffect(() => {
    retryFunction();
    if (ignoreCurrentsearch) setignoreCurrentsearch(false);
  }, [ignoreCurrentsearch]);

  return (
    <div className={styles.container}>
      <div className={styles.properties}>
        <ResultsFilteringComponent
          renderMode={renderMode}
          setRenderMode={setRenderMode}
          filters={filters}
          setFilters={(e) => {
            setFilters(e);
          }}
          setignoreCurrentsearch={setignoreCurrentsearch}
          loading={propertiesStatus.loading}
          total={totalHere}
          retrySearchFunction={retryFunction}
          defaultValues={defaultValues}
          openModal={() => setOpenModal(!openModal)}
        />
        <div
          className={styles.propertiesRowWrap}
          style={{
            flexDirection: renderMode === "card" ? "row" : "column",
            flexWrap: renderMode === "card" ? "wrap" : "nowrap",
          }}
        >
          {properties.length === 0 && !propertiesStatus.loading && (
            <div className={styles.noData}>
              <WarningRounded sx={{ fontSize: "100px" }} htmlColor="#DDDDDD" />
              <p>{t("noPropertiesFound")}</p>
            </div>
          )}
          {properties.map((d) => {
            return (
              <PropertyCard
                key={d.id}
                user={d.users}
                fulldata={d}
                type={
                  d.isSuperFeatured
                    ? "super"
                    : d.isFeatured
                    ? "simple"
                    : "original"
                }
                title={d.titulo}
                totalBeds={d?.habitaciones ?? 0}
                totalBaths={d?.baños ?? 0}
                totalHomes={Number(d?.pisos) ?? 1}
                totalParkings={d?.parking ?? 0}
                price={d.precio}
                categoy={
                  d.categories
                    ? d.categories.nombre === "Venta"
                      ? "sale"
                      : "rent"
                    : "rent"
                }
                description={d.descripcion}
                img={d.imgs && d.imgs.length > 0 ? d.imgs[0] : null}
                userImg={d.users?.img ?? null}
                userTotalProperties={d.users?.totalProperties ?? 0}
                userId={d.users?.id ?? ""}
                slug={d.slug}
                mode={renderMode === "card" ? "vertical" : "horizontal"}
                minWidth={cardsMinWidth}
              />
            );
          })}
        </div>

        {calculateTotalPages(totalHere ?? 0, 12) > 0 && (
          <div className={styles.paginationContainer}>
            <Pagination
              count={calculateTotalPages(totalHere ?? 0, 12)}
              color="primary"
              page={(filters?.offset ?? 0) + 1}
              onChange={(_, page) => {
                if (setFilters) {
                  const payload = {
                    ...filters,
                    offset: page - 1,
                  };
                  setFilters(payload);
                  retryFunction(payload);
                }
              }}
            />
            <Select
              value={filters?.limit}
              size="small"
              onChange={(event) => {
                if (setFilters) {
                  const payload = {
                    ...filters,
                    limit: event.target.value - 1,
                  };
                  setFilters(payload);
                  retryFunction(payload);
                }
              }}
            >
              {[12, 24, 32, 64, 128, 256, 512, 1024].map((p) => {
                return (
                  <MenuItem key={p} value={p}>
                    {p}
                  </MenuItem>
                );
              })}
            </Select>
          </div>
        )}
        {is693 && (
          <FiltersModal
            modalProps={{
              isOpen: openModal,
              toggle: () => setOpenModal(false),
            }}
            filters={filters}
            setFilters={setFilters}
            retrySearchFunction={retryFunction}
          />
        )}
      </div>
    </div>
  );
};

export default Results;
