/* eslint-disable @typescript-eslint/no-explicit-any */
import { FormikProps } from "formik";
import { updateProfileFormData } from "../types.d";
import { UserInterface } from "@/types/users/types";
import styles from "../styles.module.scss";
import {
  FormInputVariations,
  SelectOptionsProps,
} from "@/components/FormInput/types";
import { PRIMARY_COLOR } from "@/styles/constants";

export const initialStateUpdateProfileForm = (
  values: UserInterface
): updateProfileFormData => {
  const personalPhone = String(values?.telefonoPersonal ?? "");
  const officePhone = String(values?.telefonoOficina ?? "");

  let personalPhoneSelect = "142";
  let personalPhoneInput = "";
  let officePhoneSelect = "142";
  let officePhoneInput = "";
  let sexSelect: "O" | "M" | "F" = "O";

  if (personalPhone.length >= 10) {
    personalPhoneSelect =
      personalPhone.length > 10
        ? personalPhone.substring(0, personalPhone.length - 10)
        : "142";
    personalPhoneInput = personalPhone.substring(personalPhone.length - 10);
  }

  if (officePhone.length >= 10) {
    officePhoneSelect =
      officePhone.length > 10
        ? officePhone.substring(0, officePhone.length - 10)
        : "142";
    officePhoneInput = officePhone.substring(officePhone.length - 10);
  }

  if (values?.sex) {
    sexSelect = values.sex === "H" ? "M" : values.sex === "M" ? "F" : "O";
  }

  return {
    name: values?.nombre ?? "",
    last_name: values?.apellido ?? "",
    phone: {
      select: personalPhoneSelect,
      input: personalPhoneInput,
    },
    officePhone: {
      select: officePhoneSelect,
      input: officePhoneInput,
    },
    gender: sexSelect,
    restorePassword: 0,
    propertyName: values?.nombreInmobiliaria ?? "",
    companyProfile: values?.perfilEmpresarial ?? "",
    country: values?.country ?? "",
    state: values?.state ?? "",
    city: values?.city ?? "",
    address: values?.direccionFisica ?? "",
    promotions: false,
    tiktok: values?.tiktok ?? "",
    facebook: values?.facebookpage ?? "",
    instagram: values?.instagram ?? "",
    x: values?.twitter ?? "",
    youtube: values?.youtube ?? "",
    linkedin: values?.linkedin ?? "",
    password: "",
    confirmPassword: "",
    location: {
      lat: values?.lat ?? Number(process.env.NEXT_PUBLIC_DEFAULT_LAT),
      lng: values?.lng ?? Number(process.env.NEXT_PUBLIC_DEFAULT_LNG),
    },
    locationCenter: {
      lat: values?.lat ?? Number(process.env.NEXT_PUBLIC_CDMX_LAT) ?? 19.4326, // Latitud de CDMX (default: 19.4326)
      lng: values?.lng ?? Number(process.env.NEXT_PUBLIC_CDMX_LNG) ?? -99.1332, // Longitud de CDMX (default: -99.1332)
    },
  };
};

export const requiredFieldsUpdateProfileForm: Record<
  keyof updateProfileFormData,
  boolean
> = {
  name: true,
  last_name: true,
  phone: true,
  officePhone: false,
  gender: true,
  companyProfile: true,
  restorePassword: true,
  propertyName: false,
  country: false,
  state: false,
  city: false,
  address: false,
  promotions: false,
  tiktok: false,
  facebook: false,
  instagram: false,
  x: false,
  youtube: false,
  linkedin: false,
  password: true,
  confirmPassword: true,
  location: false,
  locationCenter: false,
};

export const keysUpdateProfileForm = [
  "header1",
  "divider",
  "name",
  "last_name",
  "phone",
  "officePhone",
  "gender",
  "restorePassword",
  "password",
  "confirmPassword",
  "companyProfile",
  "divisorFull",
  "header2",
  "propertyName",
  "country",
  "state",
  "city",
  "address",
  "map",
  "note",
  "promotions",
  "divisorFull",
  "header3",
  "tiktok",
  "facebook",
  "instagram",
  "x",
  "youtube",
  "linkedin",
  "submit",
];

export const UpdateProfileFormInputs =
  (
    phoneCodes: SelectOptionsProps[],
    countries: SelectOptionsProps[],
    states: SelectOptionsProps[],
    cities: SelectOptionsProps[],
    handleStates: (country: string, noClean?: boolean) => any,
    handleCities: (country: string, state: string) => any,
    user: UserInterface | null,
    loading?: boolean
  ) =>
  (
    key: keyof updateProfileFormData,
    formikMetadata: FormikProps<updateProfileFormData>,
    t: any
  ): Record<keyof updateProfileFormData | string, FormInputVariations> => {
    const { values, setValues, handleChange, submitForm } = formikMetadata;

    const inputs: Record<
      keyof updateProfileFormData | string,
      FormInputVariations
    > = {
      header1: {
        type: "title_and_subtitle",
        texts: {
          title: t("basicInfo"),
          showBottomDivider: true,
          titleStyle: {
            textAlign: "left",
            fontWeight: 700,
            fontSize: 20,
          },
        },
      },
      name: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      last_name: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      phone: {
        type: "select_input",
        input: {
          value: values.phone?.input,
          placeholder: t("enter10Characters"),
          onChange: (e) =>
            setValues({
              ...values,
              phone: {
                ...values.phone,
                input: e.target.value ?? "",
              },
            }),
        },
        select: {
          options: phoneCodes,
          selectedItem: values.phone?.select as string,
          placeholder: t(key),
          onClickItem: (val) =>
            setValues({
              ...values,
              phone: {
                ...values.phone,
                input: values?.phone?.input ?? "",
                select: val as string,
              },
            }),
        },
      },
      officePhone: {
        type: "select_input",
        input: {
          value: values.officePhone?.input,
          placeholder: t("enter10Characters"),
          onChange: (e) =>
            setValues({
              ...values,
              officePhone: {
                ...values.officePhone,
                input: e.target.value ?? "",
              },
            }),
        },
        select: {
          options: phoneCodes,
          selectedItem: values.officePhone?.select as string,
          placeholder: t(key),
          onClickItem: (val) =>
            setValues({
              ...values,
              officePhone: {
                ...values.officePhone,
                input: values?.officePhone?.input ?? "",
                select: val as string,
              },
            }),
        },
      },
      gender: {
        type: "selector",
        inputProps: {
          options: ["M", "F", "O"].map((d) => {
            return {
              label: t(d + "-gender"),
              name: d,
              value: d,
            };
          }),
          selectedItem: values[key] as string,
          placeholder: t(key),
          onClickItem: (val) => {
            setValues({ ...values, [key]: val });
          },
        },
      },
      restorePassword: {
        type: "button",
        label: t("restorePassword"),
        button: {
          onClick: () => {
            setValues({
              ...values,
              restorePassword: values.restorePassword ? 0 : 1,
            });
          },
          children: t("restorePassword"),
          variant: "contained",
          color: "primary",
          disableElevation: true,
          style: {
            width: "100%",
            height: "43px",
            border: "1px solid " + PRIMARY_COLOR,
            borderRadius: "7px",
            display: "flex",
            justifyContent: "center",
            alignItems: "center",
            background: "#FFF",
            fontWeight: "700",
            color: PRIMARY_COLOR,
          },
          disabled: loading,
        },
      },
      companyProfile: {
        type: "text",
        fullWidth: true,
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          multiline: true,
          style: {
            minHeight: "188px",
            alignItems: "flex-start",
            paddingTop: "8px",
          },
          onChange: handleChange(key),
        },
      },
      divisorFull: {
        type: "children",
        fullWidth: true,
        children: (
          <>
            <div className={styles.divisor} />
          </>
        ),
      },
      header2: {
        type: "title_and_subtitle",
        texts: {
          title: t("propertyData"),
          showBottomDivider: true,
          titleStyle: {
            textAlign: "left",
            fontWeight: 700,
            fontSize: 20,
          },
        },
      },
      propertyName: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      country: {
        type: "selector",
        inputProps: {
          options: countries,
          selectedItem: values[key] as string,
          placeholder: t(key),
          onClickItem: (val) => {
            handleStates(val as string);
            setValues({
              ...values,
              [key]: val,
              locationCenter: {
                lat: countries?.find((d) => d.value === val)?.lat ?? 0,
                lng: countries?.find((d) => d.value === val)?.lng ?? 0,
              },
            });
          },
        },
      },
      state: {
        type: "selector",
        inputProps: {
          options: states,
          selectedItem: values[key] as string,
          placeholder: t(key),
          onClickItem: (val) => {
            handleCities((values.country as string) ?? "", val as string);
            setValues({
              ...values,
              [key]: val,
              locationCenter: {
                lat: states?.find((d) => d.value === val)?.lat ?? 0,
                lng: states?.find((d) => d.value === val)?.lng ?? 0,
              },
            });
          },
        },
      },
      city: {
        type: "selector",
        inputProps: {
          options: cities,
          selectedItem: values[key] as string,
          placeholder: t(key),
          onClickItem: (val) => {
            setValues({
              ...values,
              [key]: val,
              locationCenter: {
                lat: cities?.find((d) => d.value === val)?.lat ?? 0,
                lng: cities?.find((d) => d.value === val)?.lng ?? 0,
              },
            });
          },
        },
      },
      address: {
        type: "text",
        fullWidth: true,
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      map: {
        type: "map",
        hideLabel: true,
        mapProps: {
          // mapContainerStyle: {
          //   height: "470px",
          //   minHeight: "470px",
          //   borderRadius: "13.42px",
          // },
          marker: {
            lat: values?.location.lat,
            lng: values?.location.lng,
          },
          center: values?.locationCenter,
          onClick: (event: google.maps.MapMouseEvent) => {
            if (event.latLng) {
              const lat = event.latLng.lat();
              const lng = event.latLng.lng();
              setValues({
                ...values,
                location: {
                  lat,
                  lng,
                },
              });
            }
          },
        },
      },
      note: {
        type: "children",
        fullWidth: true,
        children: (
          <>
            <p className={styles.note}>{t("updateprofileNote")}</p>
          </>
        ),
      },
      // promotions: {
      //   type: "radios-group",
      //   fullWidth: true,
      //   hideLabel: true,
      //   radios: [
      //     {
      //       value: "promotions",
      //       label: t("promotions"),
      //       checked: Boolean(values[key]),
      //       onClick: () => setValues({ ...values, [key]: true }),
      //     },
      //   ],
      // },
      header3: {
        type: "title_and_subtitle",
        texts: {
          title: t("socialNetworks"),
          showBottomDivider: true,
          titleStyle: {
            textAlign: "left",
            fontWeight: 700,
            fontSize: 20,
          },
        },
      },
      tiktok: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      facebook: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      instagram: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      x: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      youtube: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      linkedin: {
        type: "text",
        inputProps: {
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      },
      submit: {
        type: "button",
        button: {
          onClick: submitForm,
          children: t("updateProfile"),
          variant: "contained",
          color: "primary",
          disableElevation: true,
          style: {
            width: "254px",
            height: "50px",
            backgroundColor: PRIMARY_COLOR,
            borderRadius: "100px",
            display: "flex",
            justifyContent: "center",
            alignItems: "center",
            color: "#fff",
          },
          loading: loading && !values.restorePassword,
          disabled: loading,
        },
      },
    };

    if (values.restorePassword) {
      inputs.password = {
        type: "text",
        inputProps: {
          passwordMode: true,
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      };
      inputs.confirmPassword = {
        type: "text",
        inputProps: {
          passwordMode: true,
          value: values[key] as string,
          placeholder: t(key),
          onChange: handleChange(key),
        },
      };
    }

    return inputs;
  };
